/*
 * Copyright (C) 2010 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.chromium.ui.clipboard; 

import android.content.Context;
import android.os.Handler;
import android.os.Message;
import android.content.ClipData;
import android.content.ClipboardManagerVega;
import android.content.cbv.IClipboardPasteListener;
import android.content.cbv.IClipboardUpdatedListener;
import android.os.RemoteException;

import android.util.Log; 
                  
public class VegaClipBoardManager implements VegaClipBoardInterface {

   private static final String LOGTAG = "VegaClipBoardManager";
   private Context mContext;
   private IClipboardPasteListener mClipboardPasteListener;
   private ClipboardManagerVega mClipboardManagerVega;
   private Handler mHandler;   
   private VegaClipBoardController.ActionHandler mActionHandler;
   private static final int PASTE_CLIPDATA = 1;

   public VegaClipBoardManager(Context context, VegaClipBoardController.ActionHandler actionHandler) {
	 mContext = context;
	 mActionHandler = actionHandler;
	 mClipboardManagerVega = ((ClipboardManagerVega)mContext.getSystemService(Context.VEGA_CLIPBOARD_SERVICE));	
	 
	 mClipboardPasteListener = new IClipboardPasteListener.Stub() {
          @Override
          public void onPasteClipData(ClipData clipdata) throws RemoteException {
          Message msg = new Message();
          msg = mHandler.obtainMessage();
          msg.what = PASTE_CLIPDATA;
          msg.obj = clipdata;
          mHandler.sendMessage(msg);
          }        
       };

       mHandler = new Handler() {
          @Override 
          public void handleMessage(Message msg) {
            switch (msg.what) {
                case PASTE_CLIPDATA:{
                   ClipData clipdata = (ClipData)msg.obj;
                   for (int i=0; i<clipdata.getItemCount(); i++) {
                      CharSequence paste = clipdata.getItemAt(i).coerceToStyledText(mContext);
                      if (paste != null && mActionHandler != null) {
                           if (mClipboardManagerVega != null) {			  	
                            mActionHandler.onPasteText(paste.toString());  
                           }							
                      }
                   }                 
                }
                default:
                    super.handleMessage(msg);
            }
          }
        };		
   }

   private void setActionHandler(VegaClipBoardController.ActionHandler actionHandler) {
       mActionHandler = actionHandler;  	
   }	
   
   @Override
   public void registerVegaClipBoardPasteListener() {
       if (mClipboardManagerVega != null) {
          mClipboardManagerVega.registerClipboardPasteListener(mClipboardPasteListener);		  
       }
   }	
   
   @Override
   public void unregisterVegaClipBoardPasteListener() {
       if (mClipboardManagerVega != null) {
         mClipboardManagerVega.unregisterClipboardPasteListener(mClipboardPasteListener);	  
       } 		
   }		

   @Override
   public boolean isVegaClipBoardShowing() {
    try{   
       return (mClipboardManagerVega != null)  ? mClipboardManagerVega.isClipboardShowing() : false;     
    } catch (Exception e) {
       e.printStackTrace();    
    }    
    return false;	
   } 

   @Override
   public void closeVegaClipBoard() {
       if (mClipboardManagerVega != null) {
        unregisterVegaClipBoardPasteListener();	
        if (isVegaClipBoardShowing()) mClipboardManagerVega.closeClipBoardDialog();
       }   
   }     
	
   @Override
   public void showVegaClipBoard(boolean originalImage,  boolean enalbeImage, VegaClipBoardController.ActionHandler actionHandler) {
       setActionHandler(actionHandler);	    
       if (mClipboardManagerVega != null && !isVegaClipBoardShowing()) {  
        registerVegaClipBoardPasteListener(); 
        mClipboardManagerVega.showClipBoardDialog(originalImage, enalbeImage); 	  
       }	   
   }

   @Override
   public int getVegaClipDataSize() {
    try{   
       return (mClipboardManagerVega != null)  ? mClipboardManagerVega.getClipDataSize() : 0;     
    } catch (Exception e) {
       e.printStackTrace();    
    }    
    return 0;	
   }  

   @Override
   public boolean canVegaClipBoard() {
    try{   
       return (mClipboardManagerVega != null)  ? true: false;     
    } catch (Exception e) {
       e.printStackTrace();    
    }    
    return false;	
   } 

}
