// Copyright (c) 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef TOOLS_GN_NINJA_TARGET_WRITER_H_
#define TOOLS_GN_NINJA_TARGET_WRITER_H_

#include <iosfwd>
#include <string>

#include "base/basictypes.h"
#include "base/files/file_path.h"
#include "tools/gn/filesystem_utils.h"
#include "tools/gn/ninja_helper.h"
#include "tools/gn/path_output.h"
#include "tools/gn/settings.h"

class Target;

// Generates one target's ".ninja" file. The toplevel "build.ninja" file is
// generated by the NinjaBuildGenerator.
class NinjaTargetWriter {
 public:
  NinjaTargetWriter(const Target* target, std::ostream& out);
  ~NinjaTargetWriter();

  void Run();

  static void RunAndWriteFile(const Target* target);

 private:
  void WriteCopyRules();

  void WriteCustomRules();
  void WriteCustomArg(const std::string& arg);

  // Writs the rules for compiling each source, writing all output files
  // to the given vector.
  //
  // common_deps is a precomputed string of all ninja files that are common
  // to each build step. This is added to each one.
  void WriteCustomSourceRules(const std::string& custom_rule_name,
                              const std::string& common_deps,
                              const SourceDir& script_cd,
                              const std::string& script_cd_to_root,
                              std::vector<OutputFile>* output_files);

  void WriteCompilerVars();
  void WriteSources(std::vector<OutputFile>* object_files);
  void WriteLinkerStuff(const std::vector<OutputFile>& object_files);

  // Writes the build line for linking the target. Includes newline.
  void WriteLinkCommand(const OutputFile& external_output_file,
                        const OutputFile& internal_output_file,
                        const std::vector<OutputFile>& object_files);

  // Returns NULL if the source type should not be compiled on this target.
  const char* GetCommandForSourceType(SourceFileType type) const;

  const char* GetCommandForTargetType() const;

  const Settings* settings_;  // Non-owning.
  const Target* target_;  // Non-owning.
  std::ostream& out_;
  PathOutput path_output_;

  NinjaHelper helper_;

  DISALLOW_COPY_AND_ASSIGN(NinjaTargetWriter);
};

#endif  // TOOLS_GN_NINJA_TARGET_WRITER_H_
