// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "content/renderer/android/phone_number_detector.h"

#include <algorithm>
#include "third_party/icu/source/i18n/unicode/regex.h"
#include "base/memory/scoped_ptr.h"
#include "base/strings/string_util.h"
#include "base/strings/utf_string_conversions.h"
#include "content/public/renderer/android_content_detection_prefixes.h"
#include "net/base/escape.h"
#include "third_party/libphonenumber/src/phonenumber_api.h"
#include "third_party/libphonenumber/src/phonenumbers/phonenumbermatch.h"
#include "third_party/libphonenumber/src/phonenumbers/phonenumbermatcher.h"
#include "third_party/libphonenumber/src/phonenumbers/region_code.h"

using i18n::phonenumbers::PhoneNumberMatch;
using i18n::phonenumbers::PhoneNumberMatcher;
using i18n::phonenumbers::PhoneNumberUtil;
using i18n::phonenumbers::RegionCode;

namespace {

// Maximum number of characters to look around for phone number detection.
const size_t kMaximumTelephoneLength = 20;

const char kPhoneNumberPattern[] =
        "^(010|011|016|017|018|019|\
          02|031|032|033|041|042|044|051|052|053|054|055|061|062|063|064|\
          030|050|060|070|080|\
          001|002|003|004|005|006|007|008|009|\
          1541|1633|1655|1677|1682|\
          [14]{2,}|[15]{2,}|[16]{2,}|[18]{2,})";

const char kPhoneNumberPatternFixed_3[] =
        "(100|101|106|107|108|109|110|111|112|113|114|115|116|117|118|119|120|\
          121|122|123|124|125|126|127|128|129|182|188|131|132)";        

const char kPhoneNumberPatternFixed_4[] =
        "(1300|1301|1330|1331|1332|\
          1333|1335|1336|1337|1338|1350|1355|1357|1365|1366|1369|1372|1377|1379|\
          1382|1385|1388|1390|1397|1398|1399)";

const int kPhoneNumberPatternFixed_3_Length = 3;
const int kPhoneNumberPatternFixed_4_Length = 4;

}  // anonymous namespace

namespace content {

PhoneNumberDetector::PhoneNumberDetector()
    : region_code_(RegionCode::GetUnknown()) {
}

// Region should be empty or an ISO 3166-1 alpha-2 country code.
PhoneNumberDetector::PhoneNumberDetector(const std::string& region)
    : region_code_(region.empty() ? RegionCode::GetUnknown()
                                  : StringToUpperASCII(region)) {
}

PhoneNumberDetector::~PhoneNumberDetector() {
}

size_t PhoneNumberDetector::GetMaximumContentLength() {
  return kMaximumTelephoneLength;
}

GURL PhoneNumberDetector::GetIntentURL(const std::string& content_text) {
  if (content_text.empty())
    return GURL();

  return GURL(kPhoneNumberPrefix +
      net::EscapeQueryParamValue(content_text, true));
}

bool PhoneNumberDetector::FindContent(const string16::const_iterator& begin,
                                      const string16::const_iterator& end,
                                      size_t* start_pos,
                                      size_t* end_pos,
                                      std::string* content_text) {
  string16 utf16_input = string16(begin, end);
  std::string utf8_input = UTF16ToUTF8(utf16_input);

  PhoneNumberMatcher matcher(utf8_input, region_code_);
  if (matcher.HasNext()) {
    PhoneNumberMatch match;
    matcher.Next(&match);

    PhoneNumberUtil* phone_util = PhoneNumberUtil::GetInstance();
    phone_util->FormatNumberForMobileDialing(match.number(), region_code_,
                                             false, /* with_formatting */
                                             content_text);
    // If the number can't be dialed from the current region, the formatted
    // string will be empty.
    if (content_text->empty())
      return false;

    if (region_code_.compare("kr")) {
      int contentLength = content_text->length();
      icu::UnicodeString kPattern(kPhoneNumberPattern);
      icu::UnicodeString kInput(content_text->data(), content_text->length());
#if 0
      char* data = (char*)malloc(100);
      kInput.extract(0, contentLength, data, contentLength*4, "UTF-8");
      LOG(INFO) << "content_text : " << *content_text << ", len : " << contentLength << "kinput : "<< data;
#endif
      UErrorCode kStatus = U_ZERO_ERROR;
      icu::RegexMatcher* kMatcher;
      if (contentLength == kPhoneNumberPatternFixed_3_Length) {
        kMatcher = new icu::RegexMatcher(kPhoneNumberPatternFixed_3, kInput, UREGEX_CASE_INSENSITIVE, kStatus);
      }
      else if (contentLength == kPhoneNumberPatternFixed_4_Length) {
        kMatcher = new icu::RegexMatcher(kPhoneNumberPatternFixed_4, kInput, UREGEX_CASE_INSENSITIVE, kStatus);
      }
      else {
        kMatcher = new icu::RegexMatcher(kPattern, kInput, UREGEX_CASE_INSENSITIVE, kStatus);
      }
      
      if (!kMatcher->find()) {
        delete kMatcher;          
        return false;
      }
      delete kMatcher;
    }
    
    // Need to return start_pos and end_pos relative to a UTF16 encoding.
    *start_pos = UTF8ToUTF16(utf8_input.substr(0, match.start())).length();
    *end_pos = *start_pos + UTF8ToUTF16(match.raw_string()).length();

    return true;
  }

  return false;
}

}  // namespace content
