// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package org.chromium.content.browser.input;

import android.view.View;
import android.app.Activity;
import android.app.AlertDialog;
import android.app.SearchManager;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.content.pm.PackageManager.NameNotFoundException;
import android.content.res.TypedArray;
import android.graphics.drawable.Drawable;
import android.view.View.MeasureSpec;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import android.view.ViewGroup.LayoutParams;
import android.widget.TextView;
import android.widget.PopupWindow;
import android.widget.LinearLayout;
import android.view.LayoutInflater;
import android.util.DisplayMetrics;
import android.widget.HorizontalScrollView;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.BitmapDrawable;
import android.content.ClipboardManager;
import android.provider.Browser;
import android.text.TextUtils;
import android.widget.Toast;
import org.chromium.ui.clipboard.VegaClipBoardInterface; 
import org.chromium.ui.clipboard.VegaClipBoardController;
import org.chromium.content.R;
import com.google.common.annotations.VisibleForTesting;

import org.chromium.content.browser.PositionObserver;
import java.lang.reflect.Method;
import java.lang.reflect.Field;

/**
 * CursorController for selecting a range of text.
 */
public abstract class SelectionHandleController implements CursorController
                                                           , VegaClipBoardInterface 
{
    public interface ActionHandler {
        /**
         * Perform a select all action.
         * @return true iff the action was successful.
         */
        boolean selectAll();

        /**
         * Perform a copy (to clipboard) action.
         * @return true iff the action was successful.
         */
        boolean copy();

        /**
         * Perform a cut (to clipboard) action.
         * @return true iff the action was successful.
         */
        boolean cut();

        /**
         * Perform a paste action.
         * @return true iff the action was successful.
         */
        boolean paste();

        /**
         * @return true iff the current selection is editable (e.g. text within an input field).
         */
        boolean isSelectionEditable();

        /**
         * @return the currently selected text String.
         */
        String getSelectedText();

        boolean pasteText(String text);

        void onHide(); 	

        void onShowFindDialog(String selection);

        int getLineHeight();       
    }
 
    // The following constants match the ones in
    // third_party/WebKit/public/web/WebTextDirection.h
    private static final int TEXT_DIRECTION_DEFAULT = 0;
    private static final int TEXT_DIRECTION_LTR = 1;
    private static final int TEXT_DIRECTION_RTL = 2;

    /** The cursor controller images, lazily created when shown. */
    private HandleView mStartHandle, mEndHandle;

    /** Whether handles should show automatically when text is selected. */
    private boolean mAllowAutomaticShowing = true;

    /** Whether selection anchors are active. */
    private boolean mIsShowing;

    private View mParent;

    private int mFixedHandleX;
    private int mFixedHandleY;

    private PositionObserver mPositionObserver;
    private SelectionPopupMenu mSelectionPopupWindow;	
    private Context mContext;
    private ActionHandler mActionHandler;	
    VegaClipBoardController.ActionHandler mClipBoardActionHandler = null;	

    private static final String AOT_DIC_SERVICE_PACKAGE         = "com.pantech.app.aotdictionary";
    private static final String E_DICTIONARY_PACKAGE            = "com.diotek.diodict3.phone.pantech.skydict";

    private static final int UNINSTALL_DICT = 1;
    private static final int DISABLE_DICT = 2;
    private static final int INSTALLED_ENABLE_DICT = 3;
    
    private int mStateTranslateDict = 0;
    private int mStateTranslateAotDict = 0;

    /* check aot */
    private static final int INVALID_FLAG = -1;
    private Method mMethodGetAOTWindowType = null;
    private int mFlagAlwaysOnTop = INVALID_FLAG;
    
    public SelectionHandleController(View parent, PositionObserver positionObserver) {
        mParent = parent;
        mPositionObserver = positionObserver;
        mContext = parent.getContext();	
        mClipBoardActionHandler =  new VegaClipBoardController.ActionHandler() {
            @Override
            public boolean onPasteText(String text) {
              mActionHandler.pasteText(text); 
              return true;
            }                   
          };	
    }

    /** Automatically show selection anchors when text is selected. */
    public void allowAutomaticShowing() {
        mAllowAutomaticShowing = true;
    }

    /** Hide selection anchors, and don't automatically show them. */
    public void hideAndDisallowAutomaticShowing() {
        hide();
        mAllowAutomaticShowing = false;
    }

    @Override
    public boolean isShowing() {
        return mIsShowing;
    }

    @Override
    public void hide() {
        if (mIsShowing) {
            if (mStartHandle != null) mStartHandle.hide();
            if (mEndHandle != null) mEndHandle.hide();
            if (!isVegaClipBoardShowing()) unregisterVegaClipBoardPasteListener(); 
            mIsShowing = false;
        }
    }

    void cancelFadeOutAnimation() {
        hide();
    }

    /**
     * Updates the selection for a movement of the given handle (which
     * should be the start handle or end handle) to coordinates x,y.
     * Note that this will not actually result in the handle moving to (x,y):
     * selectBetweenCoordinates(x1,y1,x2,y2) will trigger the selection and set the
     * actual coordinates later via set[Start|End]HandlePosition.
     */
    @Override
    public void updatePosition(HandleView handle, int x, int y) {
        selectBetweenCoordinates(mFixedHandleX, mFixedHandleY, x, y);
    }

    @Override
    public void beforeStartUpdatingPosition(HandleView handle) {
        HandleView fixedHandle = (handle == mStartHandle) ? mEndHandle : mStartHandle;
        mFixedHandleX = fixedHandle.getAdjustedPositionX();
        mFixedHandleY = fixedHandle.getLineAdjustedPositionY();
    }

    /**
     * The concrete implementation must trigger a selection between the given
     * coordinates and (possibly asynchronously) set the actual handle positions
     * after the selection is made via set[Start|End]HandlePosition.
     */
    protected abstract void selectBetweenCoordinates(int x1, int y1, int x2, int y2);

    /**
     * @return true iff this controller is being used to move the selection start.
     */
    boolean isSelectionStartDragged() {
        return mStartHandle != null && mStartHandle.isDragging();
    }

    /**
     * @return true iff this controller is being used to drag either the selection start or end.
     */
    public boolean isDragging() {
        return (mStartHandle != null && mStartHandle.isDragging()) ||
               (mEndHandle != null && mEndHandle.isDragging());
    }

    @Override
    public void onTouchModeChanged(boolean isInTouchMode) {
        if (!isInTouchMode) {
            hide();
        }
    }

    @Override
    public void onDetached() {}

    /**
     * Moves the start handle so that it points at the given coordinates.
     * @param x The start handle position X in physical pixels.
     * @param y The start handle position Y in physical pixels.
     */
    public void setStartHandlePosition(float x, float y) {
        mStartHandle.positionAt((int) x, (int) y);
        if (mSelectionPopupWindow != null) {	
           mSelectionPopupWindow.positionAtCursor();	
        } 
    }

    /**
     * Moves the end handle so that it points at the given coordinates.
     * @param x The end handle position X in physical pixels.
     * @param y The end handle position Y in physical pixels.
     */
    public void setEndHandlePosition(float x, float y) {
        mEndHandle.positionAt((int) x, (int) y);
        if (mSelectionPopupWindow != null) {	
           mSelectionPopupWindow.positionAtCursor();	
        } 
    }

    /**
     * If the handles are not visible, sets their visibility to View.VISIBLE and begins fading them
     * in.
     */
    public void beginHandleFadeIn() {
        mStartHandle.beginFadeIn();
        mEndHandle.beginFadeIn();
    }

    /**
     * Sets the start and end handles to the given visibility.
     */
    public void setHandleVisibility(int visibility) {
        mStartHandle.setVisibility(visibility);
        mEndHandle.setVisibility(visibility);
    }

    /**
     * Shows the handles if allowed.
     *
     * @param startDir Direction (left/right) of start handle.
     * @param endDir Direction (left/right) of end handle.
     */
    public void onSelectionChanged(int startDir, int endDir) {
        if (mAllowAutomaticShowing) {
            showHandles(startDir, endDir);
        }
    }

    /**
     * Sets both start and end position and show the handles.
     * Note: this method does not trigger a selection, see
     * selectBetweenCoordinates()
     *
     * @param startDir Direction (left/right) of start handle.
     * @param endDir Direction (left/right) of end handle.
     */
    public void showHandles(int startDir, int endDir) {
        createHandlesIfNeeded(startDir, endDir);
        showHandlesIfNeeded();
    }

    public void showSelectPopupMenu(ActionHandler actionHandler) {
        mActionHandler = actionHandler;         		
        showSelectPopupWindow();
    }

    public void hideSelectPopupMenu() {
         if (mSelectionPopupWindow != null) {
             mSelectionPopupWindow.hide(true);
         }
    }	
    
    public void hideSelectPopupMenu(boolean unselect) {
        if (mSelectionPopupWindow != null) {
            mSelectionPopupWindow.hide(unselect);
        }
   }	

    public void showSelectPopupWindow() {
         if (!mIsShowing)	return;
         if (mSelectionPopupWindow == null) {
             mSelectionPopupWindow = new SelectionPopupMenu();
         }
         mSelectionPopupWindow.show();
    }	
 
    @VisibleForTesting
    public HandleView getStartHandleViewForTest() {
        return mStartHandle;
    }

    @VisibleForTesting
    public HandleView getEndHandleViewForTest() {
        return mEndHandle;
    }

    private void createHandlesIfNeeded(int startDir, int endDir) {
        if (mStartHandle == null) {
            mStartHandle = new HandleView(this,
                    startDir == TEXT_DIRECTION_RTL ? HandleView.RIGHT : HandleView.LEFT, mParent,
                    mPositionObserver);
        }
        if (mEndHandle == null) {
            mEndHandle = new HandleView(this,
                    endDir == TEXT_DIRECTION_RTL ? HandleView.LEFT : HandleView.RIGHT, mParent,
                    mPositionObserver);
        }
    }

    private void showHandlesIfNeeded() {
        if (!mIsShowing) {
            mIsShowing = true;
            mStartHandle.show();
            mEndHandle.show();
            setHandleVisibility(HandleView.VISIBLE);
        }
    }
    boolean canPaste() {
        return ((ClipboardManager)mContext.getSystemService(
                Context.CLIPBOARD_SERVICE)).hasPrimaryClip();
    }

    boolean isTextSelected() {
        return (mActionHandler != null && mActionHandler.getSelectedText() != null) ? true : false; 
    }

    boolean isTranslate() {
        mStateTranslateDict = 0;
        mStateTranslateAotDict = 0;
        ApplicationInfo appInfoDict = null;
        ApplicationInfo appInfoAotDict = null;
        
        try {
          appInfoDict = mContext.getPackageManager().getApplicationInfo(E_DICTIONARY_PACKAGE, PackageManager.GET_UNINSTALLED_PACKAGES | PackageManager.GET_DISABLED_COMPONENTS);
        } catch (NameNotFoundException e) {
            mStateTranslateDict = UNINSTALL_DICT;
            return false;
        }

        if (appInfoDict == null ) {
            mStateTranslateDict = UNINSTALL_DICT;
          return false;
        }

        if (!appInfoDict.enabled) {
            mStateTranslateDict =  DISABLE_DICT;
          return false;
        }

        mStateTranslateDict =  INSTALLED_ENABLE_DICT;

        try {
          appInfoAotDict = mContext.getPackageManager().getApplicationInfo(AOT_DIC_SERVICE_PACKAGE, PackageManager.GET_UNINSTALLED_PACKAGES | PackageManager.GET_DISABLED_COMPONENTS);
        } catch (NameNotFoundException e) {
            mStateTranslateAotDict = UNINSTALL_DICT;
            return false;
        }

        if (appInfoAotDict == null ) {
            mStateTranslateAotDict = UNINSTALL_DICT;
          return false;
        }

        if (!appInfoAotDict.enabled) {
            mStateTranslateAotDict =  DISABLE_DICT;
          return false;
        }

        mStateTranslateAotDict =  INSTALLED_ENABLE_DICT;
        return true;
    }

    private void installTranslate(boolean isAot) {
        AlertDialog.Builder alert = new AlertDialog.Builder(mContext);
        LinearLayout layout_= new LinearLayout(mContext);
        layout_.setOrientation(1);
        
        TextView info = new TextView(mContext);
        info.setText(R.string.selectpopup_translate_delete);
        TextView goDownload = new TextView(mContext);
        goDownload.setText(R.string.selectpopup_translate_download);
        goDownload.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                String INTENT_KEY_PACKAGENAME ="PACKAGENAME";
                Intent i = new Intent();
                i.setAction("com.pantech.app.appsplay.action.DETAILVIEW");
                i.putExtra(INTENT_KEY_PACKAGENAME, E_DICTIONARY_PACKAGE);
                i.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                mContext.startActivity(i);
            }
        });
        TextView lineBreakF = new TextView(mContext);
        lineBreakF.setText(" ");
        TextView lineBreakL = new TextView(mContext);
        lineBreakL.setText(" ");
        
        layout_.addView(info);
        layout_.addView(lineBreakF);
        layout_.addView(goDownload);
        layout_.addView(lineBreakL);
        alert.setView(layout_);
       
        alert.setTitle(R.string.selectpopup_translate);
        alert.setPositiveButton(com.android.internal.R.string.ok, new DialogInterface.OnClickListener() {             
            public void onClick(DialogInterface dialog, int which) {              
            }
        });
        alert.show();
    }
    private void enableTranslate(boolean isAot) {
        new AlertDialog.Builder(mContext)
                .setTitle(R.string.selectpopup_translate)
                .setMessage(R.string.selectpopup_translate_disable)
                .setPositiveButton(com.android.internal.R.string.ok,
                    new DialogInterface.OnClickListener() {
                        public void onClick(DialogInterface dialog, int which) {                                    
                        }
                }).show();
    }
    
    private boolean supportTranslate() {
        try {
            Class classLayoutParams = Class.forName("android.view.WindowManager$LayoutParams");
            Field fieldAlwaysOnTop = classLayoutParams.getField("TYPE_ALWAYS_ON_TOP");
            mFlagAlwaysOnTop = fieldAlwaysOnTop.getInt(classLayoutParams); 
            
            Class classViewRootImpl = Class.forName("android.view.ViewRootImpl");
            mMethodGetAOTWindowType = classViewRootImpl.getMethod("getAOTWindowType");
        } catch (Throwable e) {return false;}

        int windowType = INVALID_FLAG;
        if (mMethodGetAOTWindowType != null) {
            try {
             Integer i = (Integer)mMethodGetAOTWindowType.invoke(
                    mParent.getViewRootImpl());
             windowType = i.intValue();
            } catch (Throwable e) {return false;}
        } 

        if (mFlagAlwaysOnTop != INVALID_FLAG &&
            windowType == mFlagAlwaysOnTop) {
            return false;
        }
        return true;
    }
    
    private void translateException() {
        if (!supportTranslate()) {
            if (mStateTranslateDict == UNINSTALL_DICT || mStateTranslateAotDict == UNINSTALL_DICT) {
                Toast.makeText(mContext, R.string.selectpopup_translate_delete,  Toast.LENGTH_SHORT).show();
            }
            else if (mStateTranslateDict == DISABLE_DICT || mStateTranslateAotDict == DISABLE_DICT) {
                Toast.makeText(mContext, R.string.selectpopup_translate_disable,  Toast.LENGTH_SHORT).show();            
            }
            return;
        }
        if (mStateTranslateDict == UNINSTALL_DICT) {
            installTranslate(false);
            return;
        }
        else if (mStateTranslateDict == DISABLE_DICT) {
            enableTranslate(false);
            return;
        }

        if (mStateTranslateAotDict == UNINSTALL_DICT) {
            installTranslate(true);
            return;
        }
        else if (mStateTranslateAotDict == DISABLE_DICT) {
            enableTranslate(true);
            return;
        }
        
        mStateTranslateDict = 0;
        mStateTranslateAotDict = 0;
    }

    /*
     * This class is based on TextView.PastePopupMenu.
     */
    class SelectionPopupMenu implements OnClickListener {
        //private final PopupWindow mContainer;
        private int mPositionX;
        private int mPositionY;

        private final PopupWindow mContainer;

        private ViewGroup mContentView;
        private LinearLayout mLinearLayout;
        private ViewGroup.LayoutParams mLayoutParams = null;

        private TextView mPasteTextView;
        private TextView mAllTextView;
        private TextView mCutTextView;
        private TextView mCopyTextView;
        private TextView mFindTextView;
        private TextView mWebSearchTextView;
        private TextView mTranslateTextView;
        private TextView mShareTextView;
        private TextView mClipBoardTextView;		
		
        private boolean mTextSelected;
		
        private int mCnt = 0;
        
        public static final int SELECTALL_ID = 100;
        public static final int CUT_ID = 101;
        public static final int COPY_ID = 102;
        public static final int PASTE_ID = 103;    
        public static final int FIND_ID = 104;
        public static final int WEBSEARCH_ID = 105;
        public static final int TRANSLATE_ID = 106;
        public static final int SHARE_ID = 107;   	
        public static final int CLIPBOARD_ID = 108;   		
    	
        protected static final int POPUP_TEXT_LAYOUT = com.android.internal.R.layout.pt_text_selection_popup_text;

        public SelectionPopupMenu() {
            mContainer = new PopupWindow(mContext, null,
                    android.R.attr.textSelectHandleWindowStyle);
            mContainer.setBackgroundDrawable(new BitmapDrawable());
            mContainer.setSplitTouchEnabled(true);
            mContainer.setClippingEnabled(false);

            mContainer.setWidth(ViewGroup.LayoutParams.WRAP_CONTENT);
            mContainer.setHeight(ViewGroup.LayoutParams.WRAP_CONTENT);
            mContentView = null;
            initializeView();	
        }

        private void initializeView() {
            initializeContainerLayout();		
            initializeContentView();
        } 		

        private void initializeContainerLayout() {
            HorizontalScrollView scrollView = new HorizontalScrollView(mContext);
            scrollView.setHorizontalScrollBarEnabled(false);
            scrollView.setBackgroundResource(R.drawable.text_select_popup_bg);
            mContainer.setContentView(scrollView);
        	            
            mContentView = scrollView;

            mLinearLayout = new LinearLayout(mContext);
            mLinearLayout.setOrientation(LinearLayout.HORIZONTAL);
            mLinearLayout.setDividerDrawable(mContext.getResources().getDrawable(R.drawable.text_select_divider));
            mLinearLayout.setShowDividers(LinearLayout.SHOW_DIVIDER_MIDDLE);
            mContentView.addView(mLinearLayout);        	
        }

        public void initializeContentView() {
        	mAllTextView = createSelectItem(com.android.internal.R.string.selectAll,
        			mContext.getResources().getDrawable(com.android.internal.R.drawable.pt_ic_menu_selectall_holo_dark));
        	addToContentView(mAllTextView);
        	mAllTextView.setId(SELECTALL_ID);
        	
        	mCutTextView = createSelectItem(com.android.internal.R.string.cut,
            		mContext.getResources().getDrawable(com.android.internal.R.drawable.pt_ic_menu_cut_holo_dark));
        	addToContentView(mCutTextView);
        	mCutTextView.setId(CUT_ID);
        		
        	mCopyTextView = createSelectItem(com.android.internal.R.string.copy,
            		mContext.getResources().getDrawable(com.android.internal.R.drawable.pt_ic_menu_copy_holo_dark));
        	addToContentView(mCopyTextView);
        	mCopyTextView.setId(COPY_ID);        		
        		
        	mPasteTextView = createSelectItem(com.android.internal.R.string.paste,
            		mContext.getResources().getDrawable(com.android.internal.R.drawable.pt_ic_menu_paste_holo_dark));
        	addToContentView(mPasteTextView);
        	mPasteTextView.setId(PASTE_ID);         

        	checkClipBoardViewsVisibility();

        	mTranslateTextView = createSelectItem(R.string.selectpopup_translate,
	    		mContext.getResources().getDrawable(R.drawable.ic_selectionpopup_menu_translation));
        	addToContentView(mTranslateTextView);
        	mTranslateTextView.setId(TRANSLATE_ID);

        	mShareTextView = createSelectItem(R.string.selectpopup_share,
    			mContext.getResources().getDrawable(R.drawable.ic_selectionpopup_menu_share));
        	addToContentView(mShareTextView);
        	mShareTextView.setId(SHARE_ID);    	

        	mFindTextView = createSelectItem(R.string.selectpopup_find,
       			mContext.getResources().getDrawable(R.drawable.ic_selectionpopup_menu_search));
        	addToContentView(mFindTextView);
        	mFindTextView.setId(FIND_ID);        	
    		
        	mWebSearchTextView = createSelectItem(R.string.selectpopup_websearch,
    			mContext.getResources().getDrawable(R.drawable.ic_selectionpopup_menu_websearch));
        	addToContentView(mWebSearchTextView);
        	mWebSearchTextView.setId(WEBSEARCH_ID);    			
        }		

        protected TextView createSelectItem(int textId, Drawable topIcon) {
        	LayoutInflater inflater = (LayoutInflater)mContext.
                    getSystemService(Context.LAYOUT_INFLATER_SERVICE);
        	TextView selItem = (TextView) inflater.inflate(POPUP_TEXT_LAYOUT, null);
        	selItem.setCompoundDrawablesWithIntrinsicBounds(null, topIcon, null, null);	
        	selItem.setText(textId);
        	
        	setLayoutParams(selItem);
        	selItem.setBackgroundResource(R.drawable.btn_selection_holo);	        	
        	selItem.setOnClickListener(this);
        	
        	return selItem;
        }	

        protected void addToContentView(View view) {
        	mLinearLayout.addView(view);
        }
        
        protected void setLayoutParams(View view) {
        	if(mLayoutParams == null) {
        	   mLayoutParams = new ViewGroup.LayoutParams(
        				ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.MATCH_PARENT);
        	}
        	view.setLayoutParams(mLayoutParams); 
        }

        protected void measureContent() {
           final DisplayMetrics displayMetrics = mContext.getResources().getDisplayMetrics();
            mContentView.measure(
                    View.MeasureSpec.makeMeasureSpec(displayMetrics.widthPixels,
                            View.MeasureSpec.AT_MOST),
                    View.MeasureSpec.makeMeasureSpec(displayMetrics.heightPixels,
                            View.MeasureSpec.AT_MOST));                                    
        }	

        private void checkClipBoardViewsVisibility() {
           if (canVegaClipBoard()) {	
              mClipBoardTextView = createSelectItem(R.string.insertpopup_clipboard, mContext.getResources().getDrawable(R.drawable.ic_insertionpopup_menu_clipboard));
              addToContentView(mClipBoardTextView);
              mClipBoardTextView.setId(CLIPBOARD_ID);   			  
           }			 
        }		

        private void checkViewsVisibility() {
            boolean isEditable = (mActionHandler != null && mActionHandler.isSelectionEditable()) ? true : false; 

            boolean canPaste = isEditable && canPaste();
            boolean canCut = isEditable && isTextSelected();
            boolean canCopy = isTextSelected();
            boolean canClipBoard = false;  
            try {
                canClipBoard = isEditable && (mClipBoardTextView != null && getVegaClipDataSize() > 0);						
            } catch (Exception e) {
            }
			
            if(mAllTextView != null) {
            	mAllTextView.setVisibility(View.VISIBLE);
            } else {
            	mAllTextView.setVisibility(View.GONE);
            }
            
            if(canCut && mCutTextView != null) {
            	mCutTextView.setVisibility(View.VISIBLE);
            } else {
            	mCutTextView.setVisibility(View.GONE);
            }
            
            if(canCopy && mCopyTextView != null) {
            	mCopyTextView.setVisibility(View.VISIBLE);
            } else {
            	mCopyTextView.setVisibility(View.GONE);
            }
            
            if(canPaste && mPasteTextView != null) {
            	mPasteTextView.setVisibility(View.VISIBLE);
            } else {
            	mPasteTextView.setVisibility(View.GONE);
            }

           if (mClipBoardTextView != null) {
              if(canClipBoard) {
                 mClipBoardTextView.setVisibility(View.VISIBLE);
              } else {
                 mClipBoardTextView.setVisibility(View.GONE);
              }
           }

            if(canCopy&& mTranslateTextView != null) {
            	mTranslateTextView.setVisibility(View.VISIBLE);
            } else {
            	mTranslateTextView.setVisibility(View.GONE);
            }
    	
            if(mShareTextView != null) {
            	mShareTextView.setVisibility(View.VISIBLE);
            } else {
            	mShareTextView.setVisibility(View.GONE);
            }
	
            if(!isEditable && mFindTextView != null) {
            	mFindTextView.setVisibility(View.VISIBLE);
            } else {
            	mFindTextView.setVisibility(View.GONE);
            }
		
            if(canCopy && mWebSearchTextView != null) {
            	mWebSearchTextView.setVisibility(View.VISIBLE);
            } else {
            	mWebSearchTextView.setVisibility(View.GONE);
            }
        }

        void show() {
            positionAtCursor();
        }

        void hide(boolean unselect) {
            if (mActionHandler != null && unselect == true) {
              mActionHandler.onHide();		
            }			
            dismiss();
        }
        
        void dismiss() {
            if (mContainer != null && isShowing()) {
             mContainer.dismiss();	
            }		
        }

        boolean isShowing() {
            return mContainer.isShowing();
        }

        @Override
        public void onClick(View v) {
           if (mActionHandler == null) return;
           String selection = mActionHandler.getSelectedText();

           switch(v.getId()) {
        	case SELECTALL_ID:
        		 mActionHandler.selectAll();
            	break;
        	case CUT_ID:
        		 mActionHandler.cut();
        		 if (!isVegaClipBoardShowing()) {
        		   hideAndDisallowAutomaticShowing();	
        		   hideSelectPopupMenu();	
        		 }
            	break;
        	case COPY_ID:
        		 mActionHandler.copy();
        		 if (!isVegaClipBoardShowing()) {
        		   hideAndDisallowAutomaticShowing();	
        		   hideSelectPopupMenu();	
        		 }	
            	break;
        	case PASTE_ID:
        		 mActionHandler.paste();
        		 if (!isVegaClipBoardShowing()) {
        		   hideAndDisallowAutomaticShowing();	
        		   hideSelectPopupMenu();	
        		 }	
           	break;
        	case CLIPBOARD_ID: 
        		 showVegaClipBoard(false, false, mClipBoardActionHandler);
        		 hide(false);				 
           	break;		 
        	case FIND_ID:
        		 processFindSelection(selection);
        		 hideAndDisallowAutomaticShowing();
        		 hideSelectPopupMenu();
        		 closeVegaClipBoard();				 
           	break;
        	case WEBSEARCH_ID:
        		 processWebSearchSelection(selection);
        		 hideAndDisallowAutomaticShowing();	
        		 hideSelectPopupMenu();
        		 closeVegaClipBoard();				 
            	break;
        	case TRANSLATE_ID:
        		 if (isTranslate()) {
        		    processTranslateSelection(selection);        		    
        		 }
        		 else {
        		    translateException();
        		 }
        		 hideAndDisallowAutomaticShowing();
        		 hideSelectPopupMenu();	          		 
        		 closeVegaClipBoard();				 
            	break;
        	case SHARE_ID:
        		 processShareSelection(selection);
        		 hideAndDisallowAutomaticShowing();
        		 hideSelectPopupMenu();
        		 closeVegaClipBoard();				 
           }			   
        }

       void positionAtCursor() {  
    	    if (mStartHandle == null || mEndHandle == null) return;
    	   	checkViewsVisibility();
            measureContent();
        
            View contentView = mContainer.getContentView();
            int width = contentView.getMeasuredWidth();
            int height = contentView.getMeasuredHeight();

            final int screenWidth = mContext.getResources().getDisplayMetrics().widthPixels;
            int lineHeight = mActionHandler.getLineHeight();

            mPositionX = (int) (mStartHandle.getAdjustedPositionX() - width / 2.0f);
            mPositionX = ((mPositionX + width) > screenWidth) ? 0 : mPositionX;
            mPositionY = mStartHandle.getAdjustedPositionY() - height - lineHeight;

            final int[] coords = new int[2];
            mParent.getLocationInWindow(coords);
            coords[0] += mPositionX;
            coords[1] += mPositionY;

            if (coords[1] < 0) {
                //updateContent(false);
                //Update dimensions from new view
                contentView = mContainer.getContentView();
                width = contentView.getMeasuredWidth();
                height = contentView.getMeasuredHeight();

                // Move to right hand side of insertion cursor by default. TODO RTL text.
                final Drawable handle = mStartHandle.getDrawable();
                final int handleWidth = handle.getIntrinsicWidth();
                final int handleHight = handle.getIntrinsicHeight();
                int startX = mStartHandle.getAdjustedPositionX() + width;
                int startY = mEndHandle.getAdjustedPositionY() + handleHight + lineHeight*3; 
                coords[0]  = (startX > screenWidth || coords[0] <0 ) ? 0 : coords[0];
                coords[1]  =  ((startY + height) > mParent.getMeasuredHeight() || startY < 0) ? mParent.getMeasuredHeight()/2 : startY;
            } else {
                // Horizontal clipping
            	coords[0]  = ((mStartHandle.getAdjustedPositionX() + width) > screenWidth || coords[0] <0) ? 0 : coords[0];
            }	
            
            if ((mStartHandle.getAdjustedPositionY() < 0 && mEndHandle.getAdjustedPositionY() < 0 && mEndHandle.getAdjustedPositionX() >= 0 )  || (mStartHandle.getAdjustedPositionY() > mParent.getMeasuredHeight() && mEndHandle.getAdjustedPositionY() > mParent.getMeasuredHeight())) {
              if (isShowing()) mContainer.dismiss();
            } else {              
              if (!isShowing()) {
                 mContainer.showAtLocation(mParent, Gravity.NO_GRAVITY, coords[0], coords[1]);
              } else {
                 mContainer.update(coords[0], coords[1], width, height);	
              }
            }
        }

       public void processFindSelection(String selection) {
            if (!TextUtils.isEmpty(selection) && mActionHandler != null ) {
                  mActionHandler.onShowFindDialog(selection);
            }             
       }

       public void processWebSearchSelection(String selection){
            if (!TextUtils.isEmpty(selection)) {
              if (selection.length() > 2048) {
                selection = selection.substring(0,2048);
              }
              Intent i = new Intent(Intent.ACTION_WEB_SEARCH);
              i.putExtra(SearchManager.EXTRA_NEW_SEARCH, true);
              i.putExtra(SearchManager.QUERY, selection);
              i.putExtra(Browser.EXTRA_APPLICATION_ID, mContext.getPackageName());
              i.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
              try {
                mContext.startActivity(i);
              } catch (android.content.ActivityNotFoundException ex) {
                    // If no app handles it, do nothing.
              }
            }			  
       }

       public void processShareSelection(String selection) {
            if (!TextUtils.isEmpty(selection)) {
              Intent send = new Intent(Intent.ACTION_SEND);
              send.setType("text/plain");
              send.putExtra(Intent.EXTRA_TEXT, selection);
              try {
                Intent i = Intent.createChooser(send, mContext.getString(R.string.selectpopup_share));
                i.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                mContext.startActivity(i);
               } catch (android.content.ActivityNotFoundException ex) {
                    // If no app handles it, do nothing.
               }
            }
       }

       public boolean processTranslateSelection(String selection) {
            ApplicationInfo appInfo = null;
            try {
              appInfo = mContext.getPackageManager().getApplicationInfo("com.pantech.app.aotdictionary", 0);
            } catch (NameNotFoundException e) {
              return false;
            }

            if (appInfo!= null && !appInfo.enabled) {
              return false;
            }        
		
            if (!TextUtils.isEmpty(selection)) {
              selection = selection.replaceAll("[\r\n]", " ");
              final String BREAK_POINT = " ";
              String splitStr[] = selection.split(BREAK_POINT);
              int splitLength = splitStr.length;
              for (int i = 0; i<splitLength; i++) {
                if (splitStr[i].length() > 0) {
                 Intent intent = new Intent();
                 intent.setAction("com.pantech.intent.action.DICTIONARY_AOT");
                 intent.putExtra("words_to_search", splitStr[i]);
                 mContext.sendBroadcast(intent);
                 return true;
                }
              }
            }
            return false;			
       }   
    }
}
