/*
 * Copyright (C) 2012 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.chromium.android_webview;

import org.chromium.base.ActivityStatus;
import org.chromium.base.CalledByNative;
import org.chromium.base.JNINamespace;

import android.app.ActivityManager;
import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.os.Process;
import android.util.Log;
import android.view.ViewGroup;

import java.lang.reflect.Method;
import java.lang.reflect.Field;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

@JNINamespace("android_webview::AwRenderThreadWatchdog")
public class AwRenderThreadWatchdog implements Runnable, ActivityStatus.StateListener{
    private static final String TAG = "AwRenderThreadWatchdog";
    private static final boolean LOG_ENABLED = false;

    private static final int IS_ALIVE = 100;
    private static final int TIMED_OUT = 101;
    private static final int DIALOG_CLOSED = 102;

    private static final int HEARTBEAT_PERIOD = 10 * 1000;
    private static final int TIMEOUT_PERIOD = 30 * 1000;
    private static final int SUBSEQUENT_TIMEOUT_PERIOD = 15 * 1000;

    private static final int INVALID_PID = -1;
    private static final int INVALID_FLAG = -1;

    private int mPid = INVALID_PID;
    private Handler mHandler;
    private boolean mPaused;
    private boolean mPostedDialog;
    
    private int mFlagAlwaysOnTop = INVALID_FLAG;
    private Method mMethodGetAOTWindowType;

    private Set<ViewGroup> mViews;

    private static AwRenderThreadWatchdog sInstance;

    public synchronized static AwRenderThreadWatchdog start() {
        if (sInstance == null) {
            if (LOG_ENABLED) Log.d(TAG, "start");

            sInstance = new AwRenderThreadWatchdog();
            new Thread(sInstance, "AwRenderThreadWatchdog").start();
        }
        return sInstance;
    }

    public synchronized static void registerView(ViewGroup v) {
        if (sInstance != null && v != null) {
            sInstance.addView(v);
        }
    }

    public synchronized static void unregisterView(ViewGroup v) {
        if (sInstance != null && v != null) {
            sInstance.removeView(v);
        }
    }

    @CalledByNative
    public synchronized static void heartBeat() {
        if (sInstance != null) {
            sInstance.restart();
        }
    }

    public synchronized static void onRenderProcessCreated(int pid) {
        if (sInstance != null) {
            sInstance.renderProcessCreated(pid);
        }
    }

    @Override
    public void onActivityStateChange(int state) {
        synchronized(AwRenderThreadWatchdog.class) {
            if (sInstance != null) {
                if (state == ActivityStatus.PAUSED) {
                    pauseWatchdog();
                } else if (state == ActivityStatus.RESUMED) {
                    resumeWatchdog();
                }
            }
        }
    }

    private void renderProcessCreated(int pid) {
        if (LOG_ENABLED) Log.d(TAG, "renderProcessCreated: " + pid);

        mPid = pid;
        if (!mPaused && mPid != INVALID_PID && mViews != null && mViews.size() > 0) {
            nativeSendToRenderIsAlive(mPid);
            mHandler.sendEmptyMessageDelayed(TIMED_OUT, TIMEOUT_PERIOD);
        }
    }

    private void addView(ViewGroup v) {
        if (LOG_ENABLED) Log.d(TAG, "addView: " + v);

        if (mViews == null) {
            mViews = new HashSet<ViewGroup>();
        }
        mViews.add(v);
    }

    private void removeView(ViewGroup v) {
        if (LOG_ENABLED) Log.d(TAG, "removeView: " + v);

        if (mViews == null) {
            return;
        }

        mViews.remove(v);
    }

    private void restart() {
        if (LOG_ENABLED) Log.d(TAG, "restart");

        if (mHandler == null || mPaused) {
            return;
        }

        mHandler.removeMessages(TIMED_OUT);
        mHandler.sendEmptyMessageDelayed(TIMED_OUT, TIMEOUT_PERIOD);
        mHandler.sendEmptyMessageDelayed(IS_ALIVE, HEARTBEAT_PERIOD);
    }

    private AwRenderThreadWatchdog() {
        ActivityStatus.registerStateListener(this);

        try {
            Class classLayoutParams = Class.forName("android.view.WindowManager$LayoutParams");
            Field fieldAlwaysOnTop = classLayoutParams.getField("TYPE_ALWAYS_ON_TOP");
            mFlagAlwaysOnTop = fieldAlwaysOnTop.getInt(classLayoutParams); 

            Class classViewRootImpl = Class.forName("android.view.ViewRootImpl");
            mMethodGetAOTWindowType = classViewRootImpl.getMethod("getAOTWindowType");
        } catch (Throwable e) {
            //ignore all exceptions
            if (LOG_ENABLED) Log.d(TAG, "aot: " + e);
        }
    }

    private void pauseWatchdog() {
        if (LOG_ENABLED) Log.d(TAG, "pauseWatchdog");

        mPaused = true;

        if (mHandler == null) {
            return;
        }

        mHandler.removeMessages(TIMED_OUT);
        mHandler.removeMessages(IS_ALIVE);
    }

    private void resumeWatchdog() {
        if (LOG_ENABLED) Log.d(TAG, "resumeWatchdog");

        if (!mPaused) {
            return;
        }

        mPaused = false;

        if (mHandler == null || mPostedDialog ||
                 mViews == null || mViews.size() == 0) {
            return;
        }

        nativeSendToRenderIsAlive(mPid);
        mHandler.sendEmptyMessageDelayed(TIMED_OUT, TIMEOUT_PERIOD);
    }

    private void createHandler() {
        synchronized (AwRenderThreadWatchdog.class) {
            mHandler = new Handler() {
                @Override
                public void handleMessage(Message msg) {
                    switch (msg.what) {
                    case IS_ALIVE:
                        synchronized(AwRenderThreadWatchdog.class) {
                            if (mPaused) {
                                return;
                            }

                            nativeSendToRenderIsAlive(mPid);
                        }
                        break;
                    case TIMED_OUT:
                        if (LOG_ENABLED) Log.d(TAG, "TIMED_OUT");
                        synchronized(AwRenderThreadWatchdog.class) {
                            timeout();
                        }
                        break;
                    case DIALOG_CLOSED:
                        synchronized(AwRenderThreadWatchdog.class) {
                            mPostedDialog = false;
                        }
                        break;
                    }
                }
            };
        }
    }

    private void timeout() {
        mPostedDialog = false;
        Iterator<ViewGroup> it = mViews.iterator();
        while (it.hasNext()) {
            ViewGroup activeView = it.next();
            if (activeView.getWindowToken() != null &&
                    activeView.getViewRootImpl() != null) {
                int windowType = INVALID_FLAG;

                if (mFlagAlwaysOnTop != INVALID_FLAG && 
                        mMethodGetAOTWindowType != null) {
                    try {
                         Integer i = (Integer)mMethodGetAOTWindowType.invoke(
                                activeView.getViewRootImpl());
                         windowType = i.intValue();
                    } catch (Throwable e) {
                        //ignore all exceptions
                        if (LOG_ENABLED) Log.d(TAG, "aot: " + e);
                    }
                }

                if (LOG_ENABLED) {
                    Log.d(TAG, "mFlagAlwaysOnTop: " + mFlagAlwaysOnTop);
                    Log.d(TAG, "windowType: " + windowType);
                }

                if (mFlagAlwaysOnTop == INVALID_FLAG ||
                        windowType == INVALID_FLAG ||
                        windowType != mFlagAlwaysOnTop) {
                    mPostedDialog = activeView.post(new PageNotRespondingRunnable(
                            activeView.getContext(), mHandler));
                }

                if (mPostedDialog) {
                    break;
                }
            }
        }

        if (!mPostedDialog) {
            if (LOG_ENABLED) Log.d(TAG, "no active view");
            mHandler.sendMessageDelayed(mHandler.obtainMessage(TIMED_OUT),
                    SUBSEQUENT_TIMEOUT_PERIOD);
        }
    }

    @Override
    public void run() {
        Looper.prepare();

        createHandler();

        Looper.loop();
    }

    private class PageNotRespondingRunnable implements Runnable {
        Context mContext;
        private Handler mWatchdogHandler;

        public PageNotRespondingRunnable(Context context, Handler watchdogHandler) {
            mContext = context;
            mWatchdogHandler = watchdogHandler;
        }

        private void userWaitResponse() {
            mWatchdogHandler.sendEmptyMessage(DIALOG_CLOSED);
            mWatchdogHandler.sendEmptyMessageDelayed(TIMED_OUT, SUBSEQUENT_TIMEOUT_PERIOD);
        }

        @Override
        public void run() {
            // This must run on the UI thread as it is displaying an AlertDialog.
            assert Looper.getMainLooper().getThread() == Thread.currentThread();
            new AlertDialog.Builder(mContext)
                    .setMessage(com.android.internal.R.string.webpage_unresponsive)
                    .setPositiveButton(com.android.internal.R.string.force_close,
                            new DialogInterface.OnClickListener() {
                                @Override
                                public void onClick(DialogInterface dialog, int which) {
                                    // User chose to force close.
                                    ActivityManager am = (ActivityManager)mContext.
                                            getSystemService(Context.ACTIVITY_SERVICE);
                                    try {
                                        am.forceStopPackage(mContext.getPackageName());
                                    } catch (Exception e) {
                                        Process.killProcess(Process.myPid());
                                    }
                                }
                            })
                    .setNegativeButton(com.android.internal.R.string.wait,
                            new DialogInterface.OnClickListener() {
                                @Override
                                public void onClick(DialogInterface dialog, int which) {
                                    userWaitResponse();
                                }
                            })
                    .setOnCancelListener(
                            new DialogInterface.OnCancelListener() {
                                @Override
                                public void onCancel(DialogInterface dialog) {
                                    userWaitResponse();
                                }
                            })
                    .setIconAttribute(android.R.attr.alertDialogIcon)
                    .show();
        }
    }

    private static native void nativeSendToRenderIsAlive(int pid);
}
