/**
 * The skeleton code demonstrates how to implement
 * your own Overlay class by overriding the draw
 * and onTap handlers.
 */

package com.pantech.providers.lgucallmsgsettings;

import android.content.ContentProvider;
import android.content.ContentUris;
import android.content.ContentValues;
import android.content.Context;
import android.content.UriMatcher;
import android.database.Cursor;
import android.database.SQLException;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.database.sqlite.SQLiteQueryBuilder;
import android.database.sqlite.SQLiteDatabase.CursorFactory;
import android.net.Uri;
import android.text.TextUtils;
import android.util.Log;

public class LGUCallMsgSettingsProvider extends ContentProvider {
	  
	// The underlying database 
	private SQLiteDatabase 	mSQLiteDB;

	private static final String TAG 						= "LGUCallMsgSettingsProvider.java";
	private static final String DATABASE_NAME 				= "lgucallmsgsettings.db";
	private static final int	DATABASE_VERSION 			= 1;
	private static final String DATABASE_TABLE 				= "lgucallmsgsettings";

	//Create the constants used to differentiate between the different URI requests.
	private static final int ITEMS 							= 1;
	private static final int ITEM_ID 						= 2;
	
	private static final UriMatcher uriMatcher;
	//Allocate the UriMatcher object, where a URI ending in 'display' will
	//correspond to a request for all display, and 'display? with a trailing '/[rowID]' will represent a single earthquake row.
	static 
	{
		  uriMatcher = new UriMatcher(UriMatcher.NO_MATCH);
		  uriMatcher.addURI(LGUCallMsgSettings.AUTHORITY, DATABASE_TABLE, ITEMS);
		  uriMatcher.addURI(LGUCallMsgSettings.AUTHORITY, DATABASE_TABLE + "/#", ITEM_ID);
	}
	
	@Override
	public boolean onCreate() 	
	{
		Context context = getContext();
		
		Log.v(TAG, "onCreate() "); 
	  
	    NetworkDatabaseHelper dbHelper;
	    dbHelper = new NetworkDatabaseHelper(context, DATABASE_NAME, 
														null, DATABASE_VERSION);
	    mSQLiteDB = dbHelper.getWritableDatabase();
	    return (mSQLiteDB == null) ? false : true;
	}

	@Override
	public String getType(Uri uri) 
	{
	    switch (uriMatcher.match(uri)) 
	    {
	    case ITEMS: 
	    	return "vnd.android.cursor.dir/"+ DATABASE_TABLE;
	    case ITEM_ID: 
	    	return "vnd.android.cursor.item/" + DATABASE_TABLE;      
	    default: 
	    	throw new IllegalArgumentException("Unsupported URI: " + uri);
	    }
	}

	@Override
	public Cursor query(Uri uri, String[] projection, String selection, String[] selectionArgs, String sort) 
	{
	    SQLiteQueryBuilder qb = new SQLiteQueryBuilder();
	    qb.setTables(DATABASE_TABLE);

	    // If this is a row query, limit the result set to the passed in row. 
	    switch (uriMatcher.match(uri)) 
	    {
	    case ITEM_ID: 
	    	qb.appendWhere(LGUCallMsgSettings.KEY_ID + "=" + uri.getPathSegments().get(1));
	    	break;
	    default: 
	    	break;
	    }

	    // Apply the query to the underlying database.
	    Cursor c = qb.query(mSQLiteDB, 
	                        projection, 
	                        selection, selectionArgs, 
	                        null, null, sort);

	    // Register the contexts ContentResolver to be notified if
	    // the cursor result set changes. 
	    c.setNotificationUri(getContext().getContentResolver(), uri);
	    
	    // Return a cursor to the query result.
	    return c;
	}

	@Override
	public Uri insert(Uri uri, ContentValues values)
	{
		// Insert the new row, will return the row number if successful.
	    long rowID = mSQLiteDB.insert(DATABASE_TABLE, DATABASE_TABLE, values);
	          
	    // Return a URI to the newly inserted row on success.
	    if (rowID > 0) 
	    {
	    	Uri _uri = ContentUris.withAppendedId(LGUCallMsgSettings.CONTENT_URI, rowID);
	    	getContext().getContentResolver().notifyChange(_uri, null);
	    	return _uri;
	    }
	    throw new SQLException("Failed to insert row into " + uri);
	}

	@Override
	public int delete(Uri uri, String where, String[] whereArgs)
	{
	    int count;
	    Log.d(TAG, "delete="+uriMatcher.match(uri));
	   
	
	    switch (uriMatcher.match(uri)) 
	    {
	    	case ITEMS:
	    		count = mSQLiteDB.delete(DATABASE_TABLE, where, whereArgs);
	    		break;

	    	case ITEM_ID:
	    		String segment = uri.getPathSegments().get(1);
	    		count = mSQLiteDB.delete(DATABASE_TABLE, LGUCallMsgSettings.KEY_ID + "="
	                                    + segment
	                                    + (!TextUtils.isEmpty(where) ? " AND (" 
	                                    + where + ')' : ""), whereArgs);
	    		break;

	    	default: 
	    		Log.d(TAG, "delete--default="+uriMatcher.match(uri));
	    		throw new IllegalArgumentException("Unsupported URI: " + uri);
	    }


	    getContext().getContentResolver().notifyChange(uri, null);
	    return count;
	}

	@Override
	public int update(Uri uri, ContentValues values, String where, String[] whereArgs) 
	{
	    int count;
	    switch (uriMatcher.match(uri)) 
	    {
	    case ITEMS: count = mSQLiteDB.update(DATABASE_TABLE, values, where, whereArgs);
	    	break;

	    case ITEM_ID: String segment = uri.getPathSegments().get(1);
	    	count = mSQLiteDB.update(DATABASE_TABLE, values, LGUCallMsgSettings.KEY_ID 
	                             + "=" + segment 
	                             + (!TextUtils.isEmpty(where) ? " AND (" 
	                             + where + ')' : ""), whereArgs);
	    	break;

	    default: throw new IllegalArgumentException("Unknown URI " + uri);
	    }

	    getContext().getContentResolver().notifyChange(uri, null);
	    return count;
	}
	      
	// Helper class for opening, creating, and managing database version control
	private static class NetworkDatabaseHelper extends SQLiteOpenHelper 
	{
		private static final String DATABASE_CREATE =
	      "create table " + DATABASE_TABLE + " (" 
	      + LGUCallMsgSettings.KEY_ID + " integer primary key autoincrement, "	      	      
	      + LGUCallMsgSettings.KEY_NAME + " TEXT, "
	      + LGUCallMsgSettings.KEY_VALUE + " TEXT);";
	        
	    // Helper class for managing the Earthquake database 
	    public NetworkDatabaseHelper(Context context, String name, CursorFactory factory, int version) 
	    {
	    	super(context, name, factory, 
	    			version);
	    }

	    @Override
	    public void onCreate(SQLiteDatabase db) 
	    {
	    	db.execSQL(DATABASE_CREATE);
	    	Log.v(TAG, "Database Transtion Start"); 
	    	db.beginTransaction();
	    	
			try {
				   Log.v(TAG, "Database Transaction ? " + db.inTransaction());
				   Log.v(TAG, "Database Locked by current thread ? " + db.isDbLockedByCurrentThread());
				   
				   setInitalValue(db,LGUCallMsgSettings.CALL_MSG_BLOCK, Integer.toString(LGUCallMsgSettings.CALL_MSG_BLOCK_DEFAULT));
				   				   
				   db.setTransactionSuccessful();
				} 
				catch (Exception e)  {
	    	    	 Log.v(TAG,"Transaction fail....");
	    	    } 
				finally {
	    	        db.endTransaction();
		    	}
	    }
		
	    @Override
	    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) 
	    {
	    	Log.w(TAG, "Upgrading database from version " + oldVersion + " to "
	                  + newVersion + ", which will destroy all old data");
	              
	    	db.execSQL("DROP TABLE IF EXISTS " + DATABASE_TABLE);
	    	onCreate(db);
	    }
	    
		public void setInitalValue(SQLiteDatabase db, String name, String value) 
		{
		      ContentValues values = new ContentValues();
	          values.put(LGUCallMsgSettings.KEY_NAME, name);
	          values.put(LGUCallMsgSettings.KEY_VALUE, value); 
		      db.insertOrThrow(DATABASE_TABLE, DATABASE_TABLE ,values);	     
		}

	}
}

